%A Partial Differential Equation Model of the Synthetic Genetic Edge Detector Program
%For more information, please see:
%Tabor, J.J., Salis, H.M., Simpson, Z.B., Chevalier, A.A., Levskaya, A., Marcotte, E., Voigt, C.A., & Ellington, A.D. (2009)
%A Synthetic Genetic Edge Detection Program, Cell, 137 (7): 1271-1281

%Code written by Howard Salis (2009)
%Shameless plug: Design your synthetic DNA @ http://salis.psu.edu/software

%This is the fun part to change ... 
%Read the light mask from image files. Avoid using a high-resolution image.
%filenames = {'White_Circle_Inside.tif'}; %the light mask image
%filenames = {'hitchcock_black.tif'};
%filenames = {'hitchcock_white.tif'};
filenames = {'square_mask.tif'};
%filenames = {'single_star_mask.tif'};

background = 0; %white = 255, black = 0 ... used as the initial condition when discretizing the light mask

%Biophysical Parameters:
%These get lumped together into two dimensionless units: DU_1, DU_2
rho_cells = 1e9; %generally, 1e9 to 1e12; %number of bacterial cells per cm^3 on plate
r_TI_max = 0.2; %maximum rate of transcriptional initiation (1/sec) from Plight in absence of light
beta_luxI = 10; %LuxI proteins translated per luxI mRNA per second
beta_CI = 10; %CI proteins translated per cI mRNA per second
kcat_luxI = 0.0005; %kinetic rate of production of AHL molecules per LuxI concentration per second
delta_m_luxI = log(2) / 360; %degradation constant for luxI mRNA (t_1/2 = 360 sec = 6 min)
delta_m_cI = log(2) / 360; %degradation constant for cI mRNA (t_1/2 = 360 sec = 6 min)
delta_p_luxI = log(2) / 1200; %degradation constant for luxI protein (t_1/2 = 1200 sec = 20 min)

%These are additional parameters lumped into DU_3 and DU_4
delta_p_cI = log(2) / 1200; %degradation constant for CI protein (t_1/2 = 1200 sec = 20 min)
delta_ahl = log(2) / 86400; %degradation constant for AHL molecules (t_1/2 = 86400 sec = 1 day. ***pH DEPENDENT***)

Radius = 4.25; %radius of the plate -- cm
Diffusion = 1e-6; %diffusivity of AHL  -- cm^2/sec

%Discretization Parameters
%Set to 180 for smooth concentration profiles (as seen in the paper)
%num_r = 20; %number of discretizations in the radial direction. 
%num_theta = 20; %number of discretizations in the axial direction
num_r = 180; %number of discretizations in the radial direction. 
num_theta = 180; %number of discretizations in the axial direction


%Initial Conditions
T_End = 1000000; % a long time to ensure steady-state
N = num_r * num_theta; %total number of discretizations
Xo = zeros(2*N,1); %initial condition
index = @(i,j) (i-1)*num_theta + j; %an index that maps u(i,j) coordinates to x(i) state vector. Don't change this.

%Set the times when each light mask is placed on top of the plate and end
%with TEnd. For playing around with subjecting the bacterial lawn to
%multiple light masks at different time points
Mask_Times = [0 T_End];

if (length(Mask_Times) ~= length(filenames)+1)
    disp('There''s a missing (or too many) Mask_Time!')
    break
end

%Calculate discretization of diffusion operator on radial/axial 2D coords
[D,r] = Create_Diffusion_Radial(num_r,num_theta,index);
th = linspace(0,1-1/num_theta,num_theta).*2*pi;

%external light input in 8-bit pixel intensity.
%for light variable: 0 = dark, 255 = white.
% Light mask input. This creates a simple black circle in a white
% background. Useful for testing purposes.
% light = ones(N,1) .* 255; 
% light(index(1,1:num_theta)) = 0;
% light(index(2,1:num_theta)) = 0;
% light(index(3,1:num_theta)) = 0;

%Convert the light mask image into the discretized light input
for i=1:length(filenames);
    light{i} = convertmask2light(filenames{i},background,r,th,num_r,num_theta,index);
    figure;imshow(filenames{i});title('Light Mask Image')
end

%Convert the light input to radial/axial 2D coordinates
for i=1:length(filenames)
    [X,Y,Z] = index2circle(light{i},r,th,index);
    figure;surf(X,Y,Z);title('Light Mask');title('Discretized Light Mask Input');shading interp;
end

%
%% Example "ideal" AHL/CI transfer function 
%
%load 'AHL_CI_tf_ideal_1.mat'
%load 'AHL_CI_tf_broken_1.mat'

%load 'AHL_CI_tf_ideal_3.mat'
load 'AHL_CI_tf_broken_2.mat'

x_r = reshape(AHL_coords,   size(AHL_coords,1)   * size(AHL_coords,  2), 1);
y_r = reshape(CI_coords,    size(CI_coords, 1)   * size(CI_coords,   2), 1);
f_r = reshape(betagal_vals, size(betagal_vals,1) * size(betagal_vals,2), 1);

figure;
%surf(AHL_coords, CI_coords, betagal_vals);
contour(AHL_coords, CI_coords, betagal_vals);
pcolor(AHL_coords, CI_coords, betagal_vals);
shading interp;
fig_plux_ci_tf = gcf;
h_plux_ci_tf   = gca;
colorbar;
xlabel('AHL');
ylabel('CI');
zlabel('betagal');
set(h_plux_ci_tf, 'XScale', 'log');
set(h_plux_ci_tf, 'YScale', 'log');
%title('Ideal plux/cI transfer function');
title('plux/cI transfer function');

%
% Curve fit to AHL/CI transfer function 
%
[fitresult, gof] = Create_Surface_Fit(x_r, y_r, f_r);

% visually check the surface fit
f_computed = zeros(size(AHL_coords,1),size(AHL_coords,2));

for i=1:size(AHL_coords,1)
    for j=1:size(AHL_coords,2)
        f_computed(i,j) = fitresult(AHL_coords(i,j),CI_coords(i,j));
    end
end

figure;
contour(AHL_coords, CI_coords, f_computed);
pcolor(AHL_coords, CI_coords, f_computed);
shading interp;
fig_computed_plux_ci_tf = gcf;
h_computed_plux_ci_tf = gca;
colorbar;
xlabel('AHL');
ylabel('CI');
zlabel('betagal');
title('Computed plux/cI transfer function');
set(h_computed_plux_ci_tf, 'XScale', 'log');
set(h_computed_plux_ci_tf, 'YScale', 'log');

%
%%

%Unit conversions. Molecules to microMolar
molec2uMole = 1 ./ 6.02214199e17;

%Time scale of AHL production (microMolar per time)
lambda_1 = (rho_cells * kcat_luxI.*r_TI_max .* beta_luxI ./ delta_m_luxI ./ delta_p_luxI).*molec2uMole;

%Time scale of CI production (microMolar per time)
lambda_2 = (rho_cells * r_TI_max .* beta_CI ./ delta_m_cI).*molec2uMole;

%"Dimensionless" units (actually, they're Micromolar). They would be
%dimensionless if the concentrations were also converted into dimensionless
%units, but that would confuse people.
DU_1 = Radius^2 * lambda_1' / Diffusion;
DU_2 = Radius^2 * lambda_2' / Diffusion;
DU_3 = delta_ahl * Radius^2 / Diffusion;
DU_4 = delta_p_cI * Radius^2 / Diffusion;

clear S T M1 M2 M3 M4

%Calculate the Jacobian
Jacobian = sparse(2*N,2*N);
Jacobian(1:N,:) = sparse([D - spdiags(ones(N,1).*DU_3,0,sparse(N,N)) sparse(N,N)]);
Jacobian(1+N:2*N,:) = sparse([sparse(N,N) spdiags(-ones(N,1).*DU_4,0,sparse(N,N))]);
sparsity = find(Jacobian > 0);

%OPTIONS = odeset('Jacobian',Jacobian,'RelTol',1e-3,'AbsTol',1e-8);
OPTIONS = odeset('Jacobian',Jacobian,'JPattern',sparsity,'RelTol',1e-3,'AbsTol',1e-8);

%Solve the linear PDEs
S = [];
T = [];
for i=1:length(filenames)
    [time,sol] = ode23s(@PDE_Time,[Mask_Times(i) Mask_Times(i+1)],Xo,OPTIONS,D,light{i},DU_1,DU_2,DU_3,DU_4);
    T = [T;time];
    S = [S;sol];
    Xo = sol(length(time),:);
end

TL = length(T);

maxvalue1 = max(max(S(:,1:N)));
maxvalue2 = max(max(S(:,N+1:2*N)));

%
%% Plot all end point AHL/CI data pairs unto plux/cI transfer function
%
AHL = S(TL,1:N);
CI = S(TL,N+1:2*N);
figure(fig_plux_ci_tf);
hold on;
%scatter(AHL,CI,5,'white');
scatter(AHL_fudge(AHL),CI_fudge(CI),5,'white');

%
%% Plot the solution data
%

% for now, just show last frame (originally start with frame 1)
begin_frame = TL;

figure;
for i=begin_frame:TL
    [X,Y,Z]=index2circle(S(i,1:N),r,th,index); %microMolar
    surf(X,Y,Z);title('AHL Concentration [\muM]')
    axis([-1 1 -1 1 0 maxvalue1])
    shading interp
    M1(i) = getframe;
end

figure;
for i=begin_frame:TL
    [X,Y,Z]=index2circle(S(i,N+1:2*N),r,th,index); %microMolar
    surf(X,Y,Z);title('CI Concentration [\muM]')
    axis([-1 1 -1 1 0 maxvalue2])
    shading interp
    M2(i) = getframe;
end

figure;
for i=begin_frame:TL
    %betagal = partition_func_1([S(i,1:N) S(i,1+N:2*N)],N);
    betagal = partition_func_4_fudge(S(i,1:N),S(i,1+N:2*N),fitresult);
    [X,Y,Z]=index2circle(betagal,r,th,index); %steady-state molecules per cell
    %Z = Z .* molec2uMolar;
    %Z = Z ./ max(max(Z));
    surf(X,Y,Z);title('\beta-galactosidase Concentration (molecules)')
    %axis([-1 1 -1 1 0 1]);
    axis([-1 1 -1 1 0 max(max(Z))]);
    shading interp;
    colorbar;
    M3(i) = getframe;
end

figure;
for i=begin_frame:TL
    betagal = partition_func_4_fudge(S(i,1:N),S(i,1+N:2*N),fitresult);
    [X,Y,Z]=index2circle(betagal,r,th,index); %steady-state molecules per cell
    Z = Z ./ max(max(Z));
    plot(X(find(Z > 0.8)),Y(find(Z > 0.8)),'.');title('Location of Edges (80%+ darkness)');
    axis([-1 1 -1 1])
    M4(i) = getframe;
end    
